"use client";

import type { ReactNode, FC } from 'react';
import React, { createContext, useState, useEffect } from 'react';
import { onAuthStateChanged, type User, signInWithEmailAndPassword } from 'firebase/auth';
import { auth } from '@/lib/firebase';

interface AuthContextType {
  user: User | null;
  loading: boolean;
  isAuthenticated: boolean;
  userRoleId: number | null;
  userPermissions: Record<string, boolean> | null;
  userId: number | null;
  userName: string | null;
  orderTab: boolean | null;
  login: (email: string, password: string) => Promise<void>;
}

export const AuthContext = createContext<AuthContextType>({
  user: null,
  loading: true,
  isAuthenticated: false,
  userRoleId: null,
  userPermissions: null,
  userId: null,
  userName: null,
  orderTab: null,
  login: async () => {},
});

export const AuthProvider: FC<{ children: ReactNode }> = ({ children }) => {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [userRoleId, setUserRoleId] = useState<number | null>(null);
  const [userPermissions, setUserPermissions] = useState<Record<string, boolean> | null>(null);
  const [userId, setUserId] = useState<number | null>(null);
  const [userName, setUserName] = useState<string | null>(null);
  const [orderTab, setOrderTab] = useState<boolean | null>(null);

  const login = async (email: string, password: string) => {
    try {
      await signInWithEmailAndPassword(auth, email, password);
    } catch (error) {
      console.error('Login error:', error);
      throw error;
    }
  };

  useEffect(() => {
    const unsubscribe = onAuthStateChanged(auth, async (currentUser) => {
      setUser(currentUser);
      setLoading(false);
      if (currentUser?.email) {
        try {
          const res = await fetch(`/api/postgres/user-role-permissions?email=${encodeURIComponent(currentUser.email)}`);
          if (res.ok) {
            const data = await res.json();
            setUserRoleId(data.role_id ?? null);
            setUserPermissions(data.permissions ?? null);
            setUserId(data.user_id ?? null);
            setUserName(data.uname ?? null);
            setOrderTab(data.order_tab ?? null);
          } else {
            setUserRoleId(null);
            setUserPermissions(null);
            setUserId(null);
            setUserName(null);
            setOrderTab(null);
          }
        } catch {
          setUserRoleId(null);
          setUserPermissions(null);
          setUserId(null);
          setUserName(null);
          setOrderTab(null);
        }
      } else {
        setUserRoleId(null);
        setUserPermissions(null);
        setUserId(null);
        setUserName(null);
        setOrderTab(null);
      }
    });
    return () => unsubscribe();
  }, []);

  const isAuthenticated = !!user;

  return (
    <AuthContext.Provider value={{ user, loading, isAuthenticated, userRoleId, userPermissions, userId, userName, orderTab, login }}>
      {children}
    </AuthContext.Provider>
  );
};
