import React, { useEffect, useState, useRef } from "react";
import { Bell, AlertCircle } from "lucide-react";
import { useAuth } from "../hooks/use-auth";
import { Alert, AlertDescription, AlertTitle } from "../components/ui/alert";
import { useTranslation } from 'react-i18next';

interface ThresholdNotification {
  device_id: string;
  thresholdKey: string;
  message: string;
}

const NotificationBell: React.FC = () => {
  const { user } = useAuth();
  const { t } = useTranslation();
  const [notifications, setNotifications] = useState<ThresholdNotification[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [open, setOpen] = useState(false);
  const bellRef = useRef<HTMLButtonElement>(null);
  const popoverRef = useRef<HTMLDivElement>(null);

  // Функция для перевода ключей порогов
  const getThresholdTranslation = (thresholdKey: string) => {
    const thresholdMap: Record<string, string> = {
      '80%': t('notifications.threshold80', '80% threshold reached'),
      '85%': t('notifications.threshold85', '85% threshold reached'),
      '90%': t('notifications.threshold90', '90% threshold reached'),
      '100%': t('notifications.threshold100', '100% threshold reached'),
    };
    return thresholdMap[thresholdKey] || thresholdKey;
  };

  // Функция для перевода сообщений
  const getMessageTranslation = (message: string) => {
    //console.log('NotificationBell - Original message:', message);
    
    // Динамический маппинг для любого устройства
    const deviceMatch = message.match(/Device (.+) reached (80%|85%|90%|100%) usage threshold\./);
    if (deviceMatch) {
      const [, deviceName, threshold] = deviceMatch;
      const thresholdKey = threshold as '80%' | '85%' | '90%' | '100%';
      const translationKey = `notifications.message${thresholdKey.replace('%', '')}`;
      //console.log('NotificationBell - Translation key:', translationKey);
      const translated = t(translationKey, `Device ${deviceName} has reached ${threshold} of max value`);
      //console.log('NotificationBell - Translated message:', translated);
      return translated;
    }
    
    //console.log('NotificationBell - No match found, returning original');
    return message;
  };

  useEffect(() => {
    const fetchNotifications = async () => {
      if (!user?.email) {
        setNotifications([]);
        setLoading(false);
        return;
      }
      setLoading(true);
      setError(null);
      try {
        const response = await fetch('/api/postgres/device-threshold-flags', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ email: user.email })
        });
        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || t('errors.fetchFailed', 'Failed to fetch notifications'));
        }
        const data = await response.json();
        setNotifications(data.notifications || []);
      } catch (err: any) {
        setError(err.message || t('errors.fetchFailed', 'Failed to fetch notifications'));
        setNotifications([]);
      } finally {
        setLoading(false);
      }
    };
    fetchNotifications();
  }, [user, t]);

  // Закрытие popover при клике вне
  useEffect(() => {
    if (!open) return;
    function handleClick(e: MouseEvent) {
      if (
        popoverRef.current &&
        !popoverRef.current.contains(e.target as Node) &&
        bellRef.current &&
        !bellRef.current.contains(e.target as Node)
      ) {
        setOpen(false);
      }
    }
    document.addEventListener('mousedown', handleClick);
    return () => document.removeEventListener('mousedown', handleClick);
  }, [open]);

  return (
    <div style={{ position: 'relative', display: 'inline-block' }}>
      <button
        ref={bellRef}
        onClick={() => setOpen((prev) => !prev)}
        style={{ background: 'none', border: 'none', cursor: 'pointer', position: 'relative' }}
        aria-label={t('notificationsTab.system', 'Show notifications')}
      >
        <Bell size={24} />
        {notifications.length > 0 && (
          <span style={{
            position: 'absolute',
            top: 0,
            right: 0,
            background: 'red',
            color: 'white',
            borderRadius: '50%',
            width: 16,
            height: 16,
            fontSize: 10,
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
          }}>{notifications.length}</span>
        )}
      </button>
      {open && (
        <div
          ref={popoverRef}
          style={{
            position: 'fixed',
            top: bellRef.current ? bellRef.current.getBoundingClientRect().bottom + 8 : 40,
            left: bellRef.current ? bellRef.current.getBoundingClientRect().left : 0,
            background: 'white',
            boxShadow: '0 2px 8px rgba(0,0,0,0.15)',
            borderRadius: 8,
            minWidth: 300,
            zIndex: 2000,
            maxWidth: '90vw',
          }}
        >
          <div style={{ maxHeight: 300, overflowY: 'auto', padding: 8 }}>
            {loading ? (
              <p className="text-sm text-muted-foreground text-center py-4">{t('notificationsTab.loading', 'Loading notifications...')}</p>
            ) : error ? (
              <Alert variant="destructive">
                <AlertCircle className="h-4 w-4" />
                <AlertTitle>{t('common.error', 'Error')}</AlertTitle>
                <AlertDescription>{error}</AlertDescription>
              </Alert>
            ) : notifications.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-4">{t('notificationsTab.noNotifications', 'No notifications.')}</p>
            ) : (
              notifications.map((notification, idx) => (
                <div key={notification.device_id + notification.thresholdKey + idx} style={{ borderBottom: '1px solid #eee', padding: 8 }}>
                  <strong>{notification.device_id} - {getThresholdTranslation(notification.thresholdKey)}:</strong>
                  <div>{getMessageTranslation(notification.message)}</div>
                </div>
              ))
            )}
          </div>
        </div>
      )}
    </div>
  );
};

export default NotificationBell; 