"use client";

import React, { useState, useEffect, useCallback, useMemo } from "react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import DashboardTab from "./DashboardTab";
import PaymentHistoryTab from "./PaymentHistoryTab";
import NotificationsTab from "./NotificationsTab";
import SystemDashboardTab from "../admin-panel/SystemDashboardTab";
import SystemAdminTab from "../admin-panel/SystemAdminTab";
import SystemNotificationsTab from "../admin-panel/SystemNotificationsTab";
import UserSettingsTab from "./UserSettingsTab";
import PendingOrdersTab from "../admin-panel/PendingOrdersTab";
import { useAuth } from "../hooks/use-auth";
import { getRoleTabs } from "../config/tabs";
import { Skeleton } from "../components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "../components/ui/alert";
import { AlertCircle } from "lucide-react";
import NotificationBell from "./NotificationBell";
import { requestNotificationPermission, onMessageListener } from "../lib/firebase-messaging";
import ResponsiveTabs from './ResponsiveTabs';
import { MobileNavigation } from './MobileNavigation';
import OrderTab from './OrderTab';
import AdminMainDashboard from '../admin-panel/AdminMainDashboard';

// Import SystemNotification interface (can be defined here or imported if defined globally)
// Assuming SystemNotificationsTab exports it or we define it here for clarity
export interface SystemNotification {
  id: string;
  deviceId: string;
  message: string;
  timestamp: Date;
  thresholdKey: string;
  read: boolean;
}

export interface UserNotification {
  id: string;
  deviceId: string;
  message: string;
  timestamp: Date;
  thresholdKey: string;
  read: boolean;
}

const MainDashboard: React.FC = () => {
  const { user, orderTab, userRoleId } = useAuth();
  const [roleId, setRoleId] = useState<number | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [userNotifications, setUserNotifications] = useState<UserNotification[]>([]);
  const [activeTab, setActiveTab] = useState<string>("");

  const handleClearNotifications = useCallback(() => {
      setUserNotifications([]);
  }, []);

  const addNotification = useCallback((deviceId: string, message: string, thresholdKey: string) => {
      setUserNotifications(prev => [
        ...prev,
        {
          id: `${deviceId}-${thresholdKey}-${Date.now()}-${Math.random().toString(36).substring(2, 7)}`,
          deviceId,
          message,
          thresholdKey,
          timestamp: new Date(),
          read: false
        }
      ]);
  }, []);

  // Memoize the tab content to prevent unnecessary re-renders
  const memoizedTabContent = useMemo(() => ({
    'dashboard': <DashboardTab 
      isSystem={userRoleId === 0}
    />,
    'payment-history': <PaymentHistoryTab />,
    'order': <OrderTab />, // Новый таб заказа
    'notifications': <NotificationsTab 
      isSystem={userRoleId === 0}
    />,
    'system-dashboard': <SystemDashboardTab />,
    'system-administration': <SystemAdminTab />,
    'system-notifications': <SystemNotificationsTab />,
    'admin-dashboard': <SystemDashboardTab />,
    'pending-orders': <PendingOrdersTab />,
    'settings': <UserSettingsTab />
  }), [userRoleId, userNotifications]);

  const allTabs = userRoleId !== null ? getRoleTabs(userRoleId) : [];
  // Фильтруем табы по orderTab
  const tabs = allTabs.filter(tab => tab.id !== 'order' || orderTab !== false);
  const defaultTab = tabs[0]?.id || '';

  // Debug logging
  //console.log("roleId", roleId, "error", error, "loading", loading, "user", user);
  //console.log("tabs", tabs);

  // Set default tab on first render
  useEffect(() => {
    if (!activeTab && defaultTab) setActiveTab(defaultTab);
  }, [defaultTab, activeTab]);

  // Упрощаем логику - используем userRoleId из AuthContext
  useEffect(() => {
    if (userRoleId !== null) {
      setRoleId(userRoleId);
      setLoading(false);
    }
  }, [userRoleId]);

  useEffect(() => {
    if (user?.email && typeof window !== 'undefined') {
      // Request notification permission and get FCM token
      requestNotificationPermission().then(async (token) => {
        if (token) {
          try {
            await fetch("/api/postgres/save-fcm-token", {
              method: "POST",
              headers: { "Content-Type": "application/json" },
              body: JSON.stringify({ email: user.email, fcm_token: token }),
            });
          } catch (error) {
            console.error("Failed to save FCM token:", error);
          }
        }
      }).catch((error) => {
        console.error("Failed to request notification permission:", error);
      });

      // Set up message listener
      try {
        const messageListener = onMessageListener();
        if (messageListener instanceof Promise) {
          messageListener.then((payload: any) => {
            // Handle foreground messages here
            //console.log("Received foreground message:", payload);
          }).catch((error) => {
            console.error("Failed to set up message listener:", error);
          });
        }
      } catch (error) {
        console.error("Failed to initialize message listener:", error);
      }
    }
  }, [user?.email]);

  if (loading) {
    return <Skeleton className="w-full h-[200px]" />;
  }

  if (error || roleId === null) {
    return (
      <Alert variant="destructive">
        <AlertCircle className="h-4 w-4" />
        <AlertTitle>Error</AlertTitle>
        <AlertDescription>{error || 'Role not found'}</AlertDescription>
      </Alert>
    );
  }

  if (!tabs.length) {
    return (
      <Alert variant="destructive">
        <AlertCircle className="h-4 w-4" />
        <AlertTitle>No Tabs</AlertTitle>
        <AlertDescription>No dashboard tabs available for this user/role.</AlertDescription>
      </Alert>
    );
  }

  // Логика ролей обрабатывается в dashboard/page.tsx
  // Здесь убираем дублирующую проверку роли

  return (
    <div className="w-full p-2 sm:p-4 md:p-6" style={{ position: 'relative', minHeight: '100vh' }}>
      {/* Notification Bell for role_id = 2 */}
      {userRoleId === 2 && (
        <div className="mb-2 sm:mb-4 md:mb-6">
          <div className="flex justify-end">
            <NotificationBell />
          </div>
        </div>
      )}
      <div className="w-full pb-20 md:pb-0">
        {/* Desktop Tabs - hidden on mobile */}
        <div className="hidden lg:block">
          <ResponsiveTabs
            tabs={tabs}
            activeTab={activeTab}
            setActiveTab={setActiveTab}
          />
        </div>
        
        {/* Mobile Navigation - shown on mobile and tablet */}
        <div className="lg:hidden">
          <MobileNavigation
            tabs={tabs}
            activeTab={activeTab}
            setActiveTab={setActiveTab}
          />
        </div>
        
        {/* Tab Content */}
        {tabs.map((tab) => (
          <div
            key={tab.id}
            style={{ display: activeTab === tab.id ? 'block' : 'none' }}
            className="pt-2 lg:pt-4"
          >
            {memoizedTabContent[tab.id as keyof typeof memoizedTabContent]}
          </div>
        ))}
      </div>
    </div>
  );
};

export default MainDashboard;