"use client";

import React, { useEffect, useState } from 'react';
import { useAuth } from '../hooks/use-auth';
import { useTranslation } from 'react-i18next';
import { Card, CardHeader, CardTitle, CardContent, CardDescription } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { Input } from '../components/ui/input';
import * as OrderBuilder from './OrderBuilder';
import type { OrderItem } from './OrderBuilder';

const COMPANY_ID = process.env.NEXT_PUBLIC_SUMIT_COMPANY_ID;
const API_KEY = process.env.NEXT_PUBLIC_SUMIT_TOKEN;
const API_PUBLIC_KEY = process.env.NEXT_PUBLIC_SUMIT_API_TOKEN;

const ManagerOrderTab: React.FC = () => {
  const { t } = useTranslation();
  const { user } = useAuth();
  const [userInfo, setUserInfo] = useState<any>(null);
  const [orderItems, setOrderItems] = useState<OrderItem[]>([]);
  const [incomeItems, setIncomeItems] = useState<any[]>([]);
  const [loading, setLoading] = useState(false);
  const [quantities, setQuantities] = useState<Record<number, number>>({});
  const [error, setError] = useState<string | null>(null);
  const [paymentMethod, setPaymentMethod] = useState<string | null>(null);
  const [card, setCard] = useState({
    number: '',
    expMonth: '',
    expYear: '',
    cvv: '',
    citizenId: '',
  });

  // Получаем pId и num_of_members
  useEffect(() => {
    if (user?.email) {
      fetch(`/api/postgres/users?email=${encodeURIComponent(user.email)}`)
        .then(res => res.json())
        .then(async data => {
          let num_of_members = data.num_of_members;
          if (typeof num_of_members !== 'number' && data.p_id) {
            const fam = await fetch(`/api/postgres/family-members?p_id=${encodeURIComponent(data.p_id)}`).then(r => r.json());
            num_of_members = fam.count || 1;
          }
          setUserInfo({ ...data, num_of_members });
        });
    }
  }, [user]);

  // Основная логика заказа
  useEffect(() => {
    const fetchOrder = async () => {
      if (!userInfo?.p_id || !userInfo?.num_of_members || !user?.email) return;
      setLoading(true);
      setError(null);
      try {
        const [docs, allItems, incomeList] = await Promise.all([
          OrderBuilder.getCustomerDocuments(userInfo.p_id),
          OrderBuilder.getCustomerDocuments(userInfo.p_id).then(OrderBuilder.getAllDocumentItems),
          OrderBuilder.fetchIncomeItems(),
        ]);
        setIncomeItems(incomeList);

        // Получить устройства пользователя
        const userDevicesRes = await fetch(`/api/postgres/user-device?email=${encodeURIComponent(user.email)}`);
        const userDevicesJson = await userDevicesRes.json();
        const userDeviceIds = (userDevicesJson.devices || []).map((d: any) => d.device_id);

        // Получить все устройства и их valve_state
        const allDevicesRes = await fetch('/api/postgres/all-devices');
        const allDevicesJson = await allDevicesRes.json();
        const closedValve = (allDevicesJson.devices || []).some(
          (dev: any) => userDeviceIds.includes(dev.device_name) && dev.valve_state === 'Close valve'
        );

        // Формируем заказ
        let order = OrderBuilder.buildOrder(allItems, userInfo);

        // Если клапан закрыт — добавить товар 818563155 (если его нет)
        if (closedValve && !order.some(item => item.ItemID === 818563155)) {
          order.push({ ItemID: 818563155, Quantity: 1, Editable: false });
        }

        setOrderItems(order);
        // Для Editable товаров выставляем количество по умолчанию
        const q: Record<number, number> = {};
        order.forEach((item: OrderItem) => {
          q[item.ItemID] = item.Quantity;
        });
        setQuantities(q);
      } catch (e: any) {
        setError(e.message || 'Ошибка загрузки заказа');
      }
      setLoading(false);
    };
    fetchOrder();
  }, [userInfo]);

  // Получить имя и цену по ItemID
  const getItemInfo = (id: number) => incomeItems.find((i: any) => String(i.ID) === String(id)) || {};

  // Для ItemID: 431126765: если num_of_members === 1, то 1*7, иначе num_of_members*3.5 (округлить вверх)
  const getSpecialQuantity = (itemID: number, num_of_members: number) => {
    if (itemID !== 431126765) return undefined;
    if (num_of_members === 1) return 7;
    return Math.ceil(num_of_members * 3.5);
  };

  const tokenizeCard = async () => {
    try {
      const res = await fetch('https://api.sumit.co.il/creditguy/vault/tokenizesingleuse/', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          CardNumber: card.number,
          ExpirationMonth: parseInt(card.expMonth),
          ExpirationYear: parseInt(card.expYear),
          CVV: card.cvv,
          CitizenID: card.citizenId,
          Credentials: {
            CompanyID: COMPANY_ID,
            APIPublicKey: API_PUBLIC_KEY,
          },
        }),
      });
      const json = await res.json();
      return json?.Data?.SingleUseToken;
    } catch (e: any) {
      setError(t('orderTab.tokenizeError', 'Ошибка токенизации карты'));
      return null;
    }
  };

  const handleOrder = async () => {
    setError(null);
    if (!paymentMethod) {
      setError(t('orderTab.selectPayment', 'Выберите способ оплаты'));
      return;
    }
    setLoading(true);
    try {
      // Формируем массив товаров для заказа
      const orderItemsToSend = orderItems.map(item => {
        const info = getItemInfo(item.ItemID);
        let quantity = item.ItemID === 431126765
          ? Number(getSpecialQuantity(item.ItemID, userInfo?.num_of_members || 1) ?? 0)
          : Number(quantities[item.ItemID] ?? 0);
        const unitPrice = info.Price || 0;
        const total = quantity * unitPrice;
        return {
          id: item.ItemID,
          name: info.Name || `Item ${item.ItemID}`,
          quantity: quantity,
          unit_price: unitPrice,
          total_price: total,
        };
      });

      if (orderItemsToSend.length === 0) {
        setError(t('orderTab.selectItem', 'Выберите товар'));
        setLoading(false);
        return;
      }

      const totalAmount = orderItemsToSend.reduce((sum, item) => sum + item.total_price, 0);

      //console.log('USERINFO', userInfo);
      const userId = userInfo?.user_id || userInfo?.id;
      if (!userId) {
        setError('Не удалось определить user_id для заказа');
        setLoading(false);
        return;
      }

      if (paymentMethod === 'cash' || paymentMethod === 'bank_transfer') {
        // ВРЕМЕННЫЙ ЛОГ
        console.log('ORDER REQUEST BODY', {
          user_id: userId,
          payment_method: paymentMethod,
          items: orderItemsToSend,
          total_amount: totalAmount,
        });
        // Создаем заказ с подтверждением для наличных/банковских переводов
        const res = await fetch('/api/postgres/orders', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            user_id: userId,
            payment_method: paymentMethod,
            items: orderItemsToSend,
            total_amount: totalAmount,
          }),
        });

        const json = await res.json();
        if (json.success) {
          setError(t('orderTab.orderCreated', 'Заказ создан и ожидает подтверждения администратора'));
        } else {
          throw new Error(json.error || t('orderTab.error', 'Ошибка создания заказа'));
        }
      } else if (paymentMethod === 'credit_card') {
        // Оплата картой - прямая обработка
        let token = null;
        token = await tokenizeCard();
        if (!token) throw new Error(t('orderTab.tokenizeError', 'Ошибка токенизации карты'));

        const res = await fetch('https://api.sumit.co.il/billing/payments/charge/', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            Customer: {
              Name: userInfo?.f_name || '',
              EmailAddress: userInfo?.email || '',
            },
            Items: orderItemsToSend.map(item => ({
              Item: { Name: item.name },
              Quantity: item.quantity,
              UnitPrice: item.unit_price,
            })),
            SingleUseToken: token,
            SendDocumentByEmail: true,
            VATIncluded: true,
            DocumentType: 13,
            Credentials: {
              CompanyID: COMPANY_ID,
              APIKey: API_KEY,
            },
          }),
        });
        const json = await res.json();
        if (json.Status === 0) {
          setError(t('orderTab.success', 'Заказ успешно оформлен!'));
        } else {
          throw new Error(json.UserErrorMessage || t('orderTab.error', 'Ошибка заказа'));
        }
      }
    } catch (err: any) {
      setError(err.message || t('orderTab.error', 'Ошибка заказа'));
    }
    setLoading(false);
  };

  // Получить разрешённые способы оплаты
  const allowedPaymentMethods = React.useMemo(() => {
    if (!userInfo?.payment_methods) return ['credit_card', 'bank_transfer', 'cash'];
    try {
      const arr = JSON.parse(userInfo.payment_methods);
      return Array.isArray(arr) && arr.length > 0 ? arr : ['credit_card', 'bank_transfer', 'cash'];
    } catch {
      return ['credit_card', 'bank_transfer', 'cash'];
    }
  }, [userInfo]);

  const PAYMENT_METHODS = [
    { value: 'credit_card', label: t('orderTab.creditCard', 'Кредитная карта') },
    { value: 'bank_transfer', label: t('orderTab.bankTransfer', 'Банковский перевод') },
    { value: 'cash', label: t('orderTab.cash', 'Наличные') },
  ];

  return (
    <Card className="w-full max-w-2xl mx-auto sm:max-w-full shadow-lg">
      <CardHeader>
        <CardTitle>{t('orderTab.title', 'Оформление заказа')}</CardTitle>
        <CardDescription>{t('orderTab.description', 'Выберите товар и оформите заказ.')}</CardDescription>
      </CardHeader>
      <CardContent>
        {loading && <div>{t('common.loading', 'Загрузка...')}</div>}
        {error && <div style={{ color: 'red' }}>{error}</div>}
        <form className="shadow-lg p-4 rounded-lg bg-white">
          {orderItems.map((item: OrderItem) => {
            const info = getItemInfo(item.ItemID);
            let quantity = item.ItemID === 431126765
              ? Number(getSpecialQuantity(item.ItemID, userInfo?.num_of_members || 1) ?? 0)
              : Number(quantities[item.ItemID] ?? 0);
            const unitPrice = info.Price || 0;
            const total = quantity * unitPrice;
            return (
              <div key={item.ItemID} style={{ marginBottom: 16, borderBottom: '1px solid #eee', paddingBottom: 8 }}>
                <div style={{ fontWeight: 600 }}>{info.Name || item.ItemID}</div>
                {info.Description && <div style={{ fontSize: 12, color: '#888' }}>{t('orderTab.description', 'Описание') + ': ' + info.Description}</div>}
                <div style={{ fontSize: 13 }}>
                  {t('orderTab.price', 'Цена')}: ₪{unitPrice} {info.Unit ? `/ ${t('orderTab.unit', info.Unit)}` : ''}
                </div>
                <div style={{ fontSize: 13 }}>
                  {t('orderTab.quantity', 'Количество')}: {item.Editable ? (
                    <Input
                      type="number"
                      min={1}
                      value={quantities[item.ItemID]}
                      disabled={!item.Editable}
                      onChange={e => setQuantities(q => ({ ...q, [item.ItemID]: Number(e.target.value) }))}
                      className="w-24 ml-2 shadow-lg"
                    />
                  ) : (
                    <span>{quantity}</span>
                  )}
                </div>
                <div style={{ fontSize: 13, fontWeight: 500 }}>
                  {t('orderTab.total', 'Сумма')}: ₪{total.toFixed(2)}
                </div>
              </div>
            );
          })}
          <div style={{ fontWeight: 700, fontSize: 16, marginTop: 16, textAlign: 'right' }}>
            {t('orderTab.totalOrder', 'Итого')}: ₪{orderItems.reduce((sum, item) => {
              const info = getItemInfo(item.ItemID);
              let quantity = item.ItemID === 431126765
                ? Number(getSpecialQuantity(item.ItemID, userInfo?.num_of_members || 1) ?? 0)
                : Number(quantities[item.ItemID] ?? 0);
              const unitPrice = info.Price || 0;
              return sum + quantity * unitPrice;
            }, 0).toFixed(2)}
          </div>
          {/* Способ оплаты */}
          <div className="mb-4">
            <label className="block font-medium mb-1">{t('orderTab.paymentMethod', 'Способ оплаты')}</label>
            <div className="flex gap-4">
              {PAYMENT_METHODS.filter(m => allowedPaymentMethods.includes(m.value)).map(method => (
                <label key={method.value}>
                  <input
                    type="radio"
                    name="paymentMethod"
                    value={method.value}
                    checked={paymentMethod === method.value}
                    onChange={() => setPaymentMethod(method.value)}
                    disabled={loading}
                  /> {method.label}
                </label>
              ))}
            </div>
          </div>
          {paymentMethod === 'creditCard' && (
            <div className="mb-4 grid grid-cols-1 sm:grid-cols-2 gap-2">
              <Input
                placeholder={t('orderTab.cardNumber', 'Номер карты')}
                value={card.number}
                onChange={e => setCard({ ...card, number: e.target.value })}
                disabled={loading}
                className="shadow-lg"
              />
              <Input
                placeholder={t('orderTab.expMonth', 'Месяц')}
                value={card.expMonth}
                onChange={e => setCard({ ...card, expMonth: e.target.value })}
                disabled={loading}
                className="shadow-lg"
              />
              <Input
                placeholder={t('orderTab.expYear', 'Год')}
                value={card.expYear}
                onChange={e => setCard({ ...card, expYear: e.target.value })}
                disabled={loading}
                className="shadow-lg"
              />
              <Input
                placeholder={t('orderTab.cvv', 'CVV')}
                value={card.cvv}
                onChange={e => setCard({ ...card, cvv: e.target.value })}
                disabled={loading}
                className="shadow-lg"
              />
              <Input
                placeholder={t('orderTab.citizenId', 'ID гражданина')}
                value={card.citizenId}
                onChange={e => setCard({ ...card, citizenId: e.target.value })}
                disabled={loading}
                className="shadow-lg"
              />
            </div>
          )}
          <Button onClick={handleOrder} disabled={loading} className="w-full mt-2 shadow-lg" style={{ background: '#0B674E', color: '#fff' }}>
            {loading ? t('common.loading', 'Загрузка...') : t('orderTab.submit', 'Оформить заказ')}
          </Button>
        </form>
      </CardContent>
    </Card>
  );
};

export default ManagerOrderTab; 