"use client";

import Link from 'next/link';
import { ShieldCheck } from 'lucide-react';
import { Button } from './ui/button';
import { useAuth } from '../hooks/use-auth';
import { signOut } from 'firebase/auth';
import { auth } from '../lib/firebase'; // Renamed to avoid conflict
import { useRouter, usePathname } from 'next/navigation';
import { useState, useEffect } from 'react';
import { useTranslation } from 'react-i18next';

export function SiteHeader() {
  const { isAuthenticated, loading, user, userName, userPermissions } = useAuth();
  const router = useRouter();
  const pathname = usePathname();
  const [currentDateTime, setCurrentDateTime] = useState<string>('');
  const { t, i18n } = useTranslation();

  // Format date and time based on user language
  const formatDateTime = () => {
    const now = new Date();
    const isHebrew = i18n.language === 'he';
    
    if (isHebrew) {
      const days = ['ראשון', 'שני', 'שלישי', 'רביעי', 'חמישי', 'שישי', 'שבת'];
      const months = ['ינואר', 'פברואר', 'מרץ', 'אפריל', 'מאי', 'יוני', 'יולי', 'אוגוסט', 'ספטמבר', 'אוקטובר', 'נובמבר', 'דצמבר'];
      
      const dayOfWeek = days[now.getDay()];
      const day = now.getDate();
      const month = months[now.getMonth()];
      const year = now.getFullYear();
      const time = now.toLocaleTimeString('he-IL', { 
        hour: 'numeric', 
        minute: '2-digit',
        hour12: false 
      });
      
      return `${dayOfWeek}, ${day} ${month} ${year}, ${time}`;
    } else {
      const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
      const months = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
      
      const dayOfWeek = days[now.getDay()];
      const day = now.getDate();
      const month = months[now.getMonth()];
      const year = now.getFullYear();
      const time = now.toLocaleTimeString('en-US', { 
        hour: 'numeric', 
        minute: '2-digit',
        hour12: true 
      });
      
      return `${dayOfWeek}, ${day} ${month} ${year}, ${time}`;
    }
  };

  // Get display name - prioritize database uname, then Firebase displayName, then email
  const getDisplayName = () => {
    if (userName) {
      return userName;
    }
    if (user?.displayName) {
      return user.displayName;
    }
    if (user?.email) {
      return user.email.split('@')[0];
    }
    return t('common.user', 'User');
  };

  useEffect(() => {
    setCurrentDateTime(formatDateTime());
    
    // Update time every minute
    const interval = setInterval(() => {
      setCurrentDateTime(formatDateTime());
    }, 60000);
    
    return () => clearInterval(interval);
  }, [i18n.language]);

  const handleLogout = async () => {
    try {
      await signOut(auth);
      router.push('/logged-out');
    } catch (error) {
      console.error("Error logging out:", error);
      // Optionally show a toast message for error
    }
  };

  // Навигационные пункты

  return (
    <header className="sticky top-0 z-50 w-full border-b" style={{ background: '#1899D3' }}>
      <div className="container flex h-16 items-center justify-between px-4 md:px-6">
        <div className="flex items-center space-x-2 md:space-x-4">
          {!loading && isAuthenticated && (
            <div className="flex flex-col">
              <span className="text-xs md:text-sm font-medium text-white">{t('auth.welcome', 'Welcome, {{name}}!', { name: getDisplayName() })}</span>
              <span className="text-xs text-white/80">{currentDateTime}</span>
            </div>
          )}
        </div>
        <div className="flex items-center space-x-2 md:space-x-4">
          {!loading && isAuthenticated && (
            <Button variant="ghost" size="sm" className="text-xs md:text-sm text-white border border-white hover:bg-white/10" onClick={handleLogout}>
              {t('common.logout', 'Logout')}
            </Button>
          )}
          {!loading && !isAuthenticated && pathname !== '/login' && (
             <Button asChild variant="ghost" size="sm" className="text-xs md:text-sm text-white border border-white hover:bg-white/10">
               <Link href="/login">{t('common.login', 'Login')}</Link>
             </Button>
          )}
        </div>
      </div>
    </header>
  );
} 