import { NextResponse } from 'next/server';
import { writeFile, readFile, mkdir } from 'fs/promises';
import path from 'path';
import { existsSync } from 'fs';

const CONFIG_FILE_PATH = path.join(process.cwd(), 'config', 'system-config.json');

// Helper function to ensure config directory exists
async function ensureConfigDirectory() {
  const configDir = path.join(process.cwd(), 'config');
  if (!existsSync(configDir)) {
    await mkdir(configDir, { recursive: true });
  }
}

// Helper function to read current config
async function readConfig() {
  try {
    // First, get current environment variables
    const currentEnv = {
      MQTT_BROKER_URL: process.env.MQTT_BROKER_URL || '',
      MQTT_APP_ID: process.env.MQTT_APP_ID || '',
      NEXT_PUBLIC_API_URL: process.env.NEXT_PUBLIC_API_URL || '',
      NEXT_PUBLIC_APP_NAME: process.env.NEXT_PUBLIC_APP_NAME || '',
      NEXT_PUBLIC_MQTT_BROKER_URL: process.env.NEXT_PUBLIC_MQTT_BROKER_URL || '',
      NEXT_PUBLIC_MQTT_APP_ID: process.env.NEXT_PUBLIC_MQTT_APP_ID || '',
    };

    // Try to read existing config file
    let existingConfig;
    try {
      const configData = await readFile(CONFIG_FILE_PATH, 'utf-8');
      existingConfig = JSON.parse(configData);
    } catch {
      // If file doesn't exist or is invalid, use default config
      existingConfig = {
        env: currentEnv,
        ui: {
          menu: {
            dashboard: 'Dashboard',
            devices: 'Devices',
            settings: 'Settings',
            profile: 'Profile',
          },
          messages: {
            success: 'Operation completed successfully',
            error: 'An error occurred',
            warning: 'Warning',
            info: 'Information',
          },
          alerts: {
            deviceUpdate: 'Device updated successfully',
            maxValueUpdate: 'Max value updated successfully',
            valveControl: 'Valve control command sent',
          },
          tabs: {
            system: 'System',
            devices: 'Devices',
            settings: 'Settings',
          },
          titles: {
            mainDashboard: 'Main Dashboard',
            deviceDetails: 'Device Details',
            systemConfig: 'System Configuration',
          },
        },
      };
    }

    // Merge current environment variables with existing config
    return {
      ...existingConfig,
      env: {
        ...currentEnv,
        ...existingConfig.env,
      },
    };
  } catch (error) {
    console.error('Error reading config:', error);
    throw error;
  }
}

export async function GET() {
  try {
    const config = await readConfig();
    return NextResponse.json(config);
  } catch (error) {
    console.error('Error reading system config:', error);
    return NextResponse.json(
      { error: 'Failed to read system configuration' },
      { status: 500 }
    );
  }
}

export async function POST(request: Request) {
  try {
    const { type, data } = await request.json();
    const currentConfig = await readConfig();

    // Update the specific section of the config
    if (type === 'env') {
      currentConfig.env = { ...currentConfig.env, ...data };
      
      // Update .env file with new values
      const envContent = Object.entries(data)
        .map(([key, value]) => `${key}=${value}`)
        .join('\n');
      
      await writeFile(path.join(process.cwd(), '.env'), envContent, 'utf-8');
    } else if (type === 'ui') {
      currentConfig.ui = { ...currentConfig.ui, ...data };
    }

    // Save updated config
    await ensureConfigDirectory();
    await writeFile(CONFIG_FILE_PATH, JSON.stringify(currentConfig, null, 2));

    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('Error updating system config:', error);
    return NextResponse.json(
      { error: 'Failed to update system configuration' },
      { status: 500 }
    );
  }
} 