import { NextRequest, NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

export async function GET(request: Request) {
  const { searchParams } = new URL(request.url);
  const email = searchParams.get('email');
  const uid = searchParams.get('uid');

  if (!email || !uid) {
    return NextResponse.json({ error: 'Missing email or uid parameter' }, { status: 400 });
  }

  try {
    // First verify the user is an admin
    const userData = await postgresClient.getUserData(email, uid);
    
    if (!userData || userData.role_id !== 1) {
      return NextResponse.json({ error: 'Unauthorized access' }, { status: 403 });
    }

    // Get admin's own devices
    const adminDevicesQuery = `
      SELECT device_id, role_id
      FROM users
      WHERE role_id = 1 AND device_id IS NOT NULL;
    `;
    
    // Get all devices
    const allDevicesQuery = `
      SELECT device_id, role_id
      FROM users
      WHERE device_id IS NOT NULL;
    `;
    
    const [adminDevicesResult, allDevicesResult] = await Promise.all([
      postgresClient.query<{ device_id: string; role_id: number }>(adminDevicesQuery),
      postgresClient.query<{ device_id: string; role_id: number }>(allDevicesQuery)
    ]);

    const adminDevices = adminDevicesResult.rows.map(row => row.device_id);
    const allDevices = allDevicesResult.rows;

    // Get max values for all devices
    const maxValues = await Promise.all(
      allDevices.map(async (device) => {
        const maxValue = await postgresClient.getDeviceMaxValue(device.device_id);
        return {
          deviceId: device.device_id,
          roleId: device.role_id,
          maxValue: maxValue !== null ? Number(maxValue) : null,
          isAdminDevice: adminDevices.includes(device.device_id)
        };
      })
    );

    return NextResponse.json({ 
      devices: maxValues,
      adminDevices: adminDevices
    });
  } catch (error: any) {
    console.error('Failed to fetch admin devices:', error);
    return NextResponse.json({ error: `Failed to retrieve devices: ${error.message || error}` }, { status: 500 });
  }
} 