"use client";

import React, { useState, useEffect, useCallback, useMemo } from "react";
import { Card, CardContent, CardHeader, CardTitle } from '../components/ui/card';
import { Skeleton } from '../components/ui/skeleton';
import { Alert, AlertDescription, AlertTitle } from '../components/ui/alert';
import { AlertCircle, Wifi, WifiOff, Server, CheckCircle2, BellOff } from "lucide-react";
import GaugeChart from './GaugeChart';
import { useAuth } from '../hooks/use-auth';
import { ResponsiveContainer, ComposedChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, AreaChart, Area } from "recharts";
import DeviceDetailModal from './DeviceDetailModal';
import { loadMqttLibrary, subscribeMqttStatus, ensureMqttConnected } from '../services/mqttService';
import { useTranslation } from "react-i18next";

type DeviceData = {
  deviceId: string;
  maxValue: number | null;
  currentRead: number | null;
  valveState: string | null;
  batteryVoltage: number | null;
  deviceType: string | null;
  threshold80?: boolean;
  threshold85?: boolean;
  threshold90?: boolean;
  threshold100?: boolean;
  roleId?: number;
};

const durationOptions = [
  { value: '1h', label: 'Last hour' },
  { value: '6h', label: 'Last 6 hours' },
  { value: '24h', label: 'Last 24 hours' },
  { value: '7d', label: 'Last 7 days' },
  { value: '30d', label: 'Last 30 days' },
  { value: '365d', label: 'Last 365 days' },
];

// System Status Card
const SystemStatusCard: React.FC<{ status: { mqttConnected: boolean; lastUpdate: Date | null; activeDevices: number; totalDevices: number; } }> = ({ status }) => {
  const { t } = useTranslation();
  return (
    <Card className="mb-6">
      <CardHeader className="pb-2">
        <CardTitle className="text-lg">{t('dashboard.systemStatus', 'System Status')}</CardTitle>
      </CardHeader>
      <CardContent>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="flex items-center space-x-8 "><div className="ml-4">
            {status.mqttConnected ? (
              <Wifi className="h-5 w-5 text-green-500" />
            ) : (
              <WifiOff className="h-5 w-5 text-red-500" />
            )}</div>
            <div className="ml-6">
              <p className="text-sm font-medium"> {t('dashboard.mqttConnection', 'MQTT Connection')}</p>
              <p className="text-sm text-muted-foreground">
                {status.mqttConnected ? t('dashboard.connected', 'Connected') : t('dashboard.disconnected', 'Disconnected')}
              </p>
            </div>
          </div>
          <div className="flex items-center space-x-8"><div className="ml-4">
            <Server className="h-5 w-5 text-blue-500" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium"> {t('dashboard.activeDevices', 'Active Devices')}</p>
              <p className="text-sm text-muted-foreground">
                {status.activeDevices} / {status.totalDevices}
              </p>
            </div>
          </div>
          <div className="flex items-center space-x-8"><div className="ml-4">
            <CheckCircle2 className="h-5 w-5 text-blue-500" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium"> {t('dashboard.lastUpdate', 'Last Update')}</p>
              <p className="text-sm text-muted-foreground">
                {status.lastUpdate ? status.lastUpdate.toLocaleTimeString() : t('dashboard.never', 'Never')}
              </p>
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

// Helper to process bar data for chart (usage difference, label formatting)
const getProcessedBarData = (rawData: { month: string; usage: number }[], deviceType: string) =>
  rawData
    .map((entry, idx, arr) => {
      if (idx === 0) return null;
      const prev = arr[idx - 1];
      let usage = entry.usage - prev.usage;
      if (usage < 0) usage = 0;
      const [mm, yyyy] = entry.month.split('_');
      // Всегда цифровой формат месяца и года
      const label = `${mm.padStart(2, '0')}.${yyyy}`;
      return { ...entry, usage, label };
    })
    .filter(Boolean);

// Горизонтальный bar chart для water/electric meter с загрузкой deviceData
const HorizontalBarChart: React.FC<{
  deviceName: string;
  devEUI: string;
  unit?: string;
  deviceType?: string;
  hideValveState?: boolean;
}> = ({ deviceName, devEUI, unit = '', deviceType, hideValveState }) => {
  const { t, i18n } = useTranslation();
  const [deviceData, setDeviceData] = React.useState<any>(null);
  const [loading, setLoading] = React.useState(true);
  const [error, setError] = React.useState<string | null>(null);

  React.useEffect(() => {
    const fetchDeviceData = async () => {
      setLoading(true);
      setError(null);
      try {
        const response = await fetch('/api/postgres/device-data', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ deviceName })
        });
        if (!response.ok) throw new Error('Failed to fetch device data');
        const data = await response.json();
        setDeviceData(data);
      } catch (err: any) {
        setError(err.message || 'Failed to fetch device data');
        setDeviceData(null);
      } finally {
        setLoading(false);
      }
    };
    if (deviceName) fetchDeviceData();
  }, [deviceName]);

  if (loading) return <div className="w-full h-16"><Skeleton className="w-full h-full" /></div>;
  if (error || !deviceData) return <div className="text-destructive text-sm">{error || t('gaugeChart.noDeviceData', 'No device data available')}</div>;

  const current = deviceData.current_read ?? 0;
  const max = deviceData.max_value ?? 0;
  const percent = max > 0 ? Math.min(100, (current / max) * 100) : 0;
  let barColor = '#4CAF50';
  if (percent >= 90) barColor = '#F44336';
  else if (percent >= 75) barColor = '#FF9800';
  else if (percent >= 50) barColor = '#FFEB3B';
  else if (percent >= 25) barColor = '#8BC34A';

  // valve state (с иконкой)
  const state = (deviceData.valve_state || '').trim().toLowerCase();
  const isWaterMeter = unit === 'm³';
  let valveIcon: string | null = null;
  let valveLabel = '';
  if (state === 'open valve') {
    valveIcon = '/img/open.png';
    valveLabel = isWaterMeter ? t('gaugeChart.open', 'Open') : t('gaugeChart.on', 'On');
  } else if (state === 'close valve') {
    valveIcon = '/img/close.png';
    valveLabel = isWaterMeter ? t('gaugeChart.close', 'Close') : t('gaugeChart.off', 'Off');
  } else {
    valveLabel = t('gaugeChart.noValve', 'No valve info');
  }

  // battery
  const battery = deviceData.battery_voltage !== undefined && deviceData.battery_voltage !== null
    ? `${deviceData.battery_voltage.toFixed(2)}V`
    : t('gaugeChart.noBattery', 'No battery info');

  const dir = i18n.dir();

  // LTR bar and ticks
  return (
    <div className="flex flex-col gap-2 w-full" dir="ltr">
      <div className="flex items-center justify-between">
        <span className="font-bold text-2xl">{current.toFixed(2)} {unit}</span>
        <span className="text-sm text-muted-foreground">{t(deviceType === 'Electric meter' ? 'gaugeChart.electricityUsage' : 'gaugeChart.waterUsage', deviceType === 'Electric meter' ? 'Electricity Usage' : 'Water Usage')}</span>
        {!hideValveState && (
          <span className="flex items-center gap-1 font-bold" style={{ color: barColor }}>
            {valveIcon && <img src={valveIcon} alt="valve" className="inline w-5 h-5 align-middle" />}
            {valveLabel}
          </span>
        )}
      </div>
      {/* Строка с процентом использования */}
      <div className="text-sm font-bold" style={{ color: barColor }} dir={dir}>
        {dir === 'rtl'
          ? <>{t('gaugeChart.percentUsedLabel', 'מנוצל')} <span>{Math.round(percent)}%</span></>
          : t('gaugeChart.percentUsed', '{{percent}}% used', { percent: Math.round(percent) })}
      </div>
      <div className="w-full h-3 bg-gray-200 rounded-full overflow-hidden relative" dir="ltr">
        <div
          className="h-full rounded-full transition-all absolute left-0 top-0"
          style={{ width: `${percent}%`, background: barColor }}
        />
      </div>
      {/* ticks */}
      <div className="flex flex-row justify-between text-xs text-muted-foreground w-full select-none" dir="ltr">
        <span>0%</span>
        <span>25%</span>
        <span>50%</span>
        <span>75%</span>
        <span>100%</span>
      </div>
      <div className="flex flex-wrap gap-4 mt-2 text-xs text-muted-foreground items-center" dir={dir}>
  {/* Максимум и батарея: всегда сначала подпись, потом значение, потом единица измерения */}
  <span>
    {t('gaugeChart.maxValue', 'Max')}: <b>{max.toFixed(2)}</b> <span dir="ltr">{unit}</span>
  </span>
  <span>
    {t('gaugeChart.battery', 'Battery')}: <b>{battery}</b>
  </span>
</div>
    </div>
  );
};

const ValveState: React.FC<{ deviceName: string; unit: string }> = ({ deviceName, unit }) => {
  const { t } = useTranslation();
  const [deviceData, setDeviceData] = React.useState<any>(null);
  const [loading, setLoading] = React.useState(true);
  const [error, setError] = React.useState<string | null>(null);

  React.useEffect(() => {
    const fetchDeviceData = async () => {
      setLoading(true);
      setError(null);
      try {
        const response = await fetch('/api/postgres/device-data', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ deviceName })
        });
        if (!response.ok) throw new Error('Failed to fetch device data');
        const data = await response.json();
        setDeviceData(data);
      } catch (err: any) {
        setError(err.message || 'Failed to fetch device data');
        setDeviceData(null);
      } finally {
        setLoading(false);
      }
    };
    if (deviceName) fetchDeviceData();
  }, [deviceName]);

  if (loading) return <Skeleton className="w-16 h-5" />;
  if (error || !deviceData) return <div className="text-destructive text-sm">{error || t('gaugeChart.noValveInfo', 'No valve info')}</div>;

  const state = (deviceData.valve_state || '').trim().toLowerCase();
  const isWaterMeter = unit === 'm³';
  let valveIcon: string | null = null;
  let valveLabel = '';
  if (state === 'open valve') {
    valveIcon = '/img/open.png';
    valveLabel = isWaterMeter ? t('gaugeChart.open', 'Open') : t('gaugeChart.on', 'On');
  } else if (state === 'close valve') {
    valveIcon = '/img/close.png';
    valveLabel = isWaterMeter ? t('gaugeChart.close', 'Close') : t('gaugeChart.off', 'Off');
  } else {
    valveLabel = t('gaugeChart.noValve', 'No valve info');
  }

  return (
    <div className="flex items-center gap-2">
      {valveIcon && <img src={valveIcon} alt="valve" className="inline w-5 h-5 align-middle" />}
      <span>{valveLabel}</span>
    </div>
  );
};

// Кастомный tooltip для usage trend
const CustomUsageTooltip = ({ active, payload, label, deviceType }: any) => {
  if (active && payload && payload.length) {
    const value = payload[0].value;
    const unit = deviceType === 'Electric meter' ? 'kW' : 'm³';
    return (
      <div className="rounded bg-background border px-3 py-2 shadow text-xs">
        <div className="mb-1 font-medium">{label}</div>
        <div>
          <span className="font-bold">{Number(value).toFixed(2)} {unit}</span>
        </div>
      </div>
    );
  }
  return null;
};

// Кастомный компонент для вертикальных меток под осью X
const VerticalTick = (props: any) => {
  const { x, y, payload } = props;
  return (
    <g transform={`translate(${x},${y + 50})`}>
      <text x={0} y={0} dy={0} dx={-8} textAnchor="end" fontSize={12} transform="rotate(-90)">
        {payload.value}
      </text>
    </g>
  );
};

const SystemDashboardTab: React.FC = () => {
  const { user, userPermissions, userRoleId } = useAuth();
  const [devices, setDevices] = useState<any[]>([]);
  const [barData, setBarData] = useState<Record<string, { month: string; usage: number }[]>>({});
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [selectedDuration, setSelectedDuration] = useState<string>('365d');
  const [notifications, setNotifications] = useState<any[]>([]);
  const [notifLoading, setNotifLoading] = useState(true);
  const [notifError, setNotifError] = useState<string | null>(null);
  // System status state
  const [systemStatus, setSystemStatus] = useState({
    mqttConnected: false, // You can wire this up to your MQTT logic if needed
    lastUpdate: null as Date | null,
    activeDevices: 0,
    totalDevices: 0,
  });
  const [allDevices, setAllDevices] = useState<any[]>([]); // all devices for the table
  const [tableLoading, setTableLoading] = useState(true);
  const [tableError, setTableError] = useState<string | null>(null);
  const [sortField, setSortField] = useState<string>('device_name');
  const [sortDirection, setSortDirection] = useState<'asc' | 'desc'>('asc');
  const [selectedDevice, setSelectedDevice] = useState<any | null>(null);
  const [modalOpen, setModalOpen] = useState(false);
  const { t } = useTranslation();
  const [search, setSearch] = useState('');
  const [selectedMobileDevice, setSelectedMobileDevice] = useState<string>('');

  // Расширенный маппинг типа устройства к ключу перевода
  const DEVICE_TYPE_MAP: Record<string, string> = {
    'waterMeter': 'devices.waterMeter',
    'electricMeter': 'devices.electricMeter',
    'Water meter': 'devices.waterMeter',
    'Electric meter': 'devices.electricMeter',
    'Water Meter': 'devices.waterMeter',
    'Electric Meter': 'devices.electricMeter',
    // Добавьте другие типы устройств по мере необходимости
  };

  const getDeviceTypeLabel = (type: string) => {
    if (!type) return '';
    const key = DEVICE_TYPE_MAP[type.trim()] || type.trim();
    return t(key, type);
  };

  // Fetch user devices and roleId
  const fetchUserDevices = useCallback(async () => {
    if (!user?.email) {
      setError('User information not available');
      setLoading(false);
      return;
    }
    setLoading(true);
    setError(null);
    try {
      const response = await fetch(`/api/postgres/user-device?email=${encodeURIComponent(user.email)}`);
      if (!response.ok) throw new Error('Failed to fetch user devices');
      const data = await response.json();
      setDevices(data.devices || []);
      setSystemStatus((prev) => ({
        ...prev,
        activeDevices: (data.devices || []).length,
        totalDevices: (data.devices || []).length,
        lastUpdate: new Date(),
      }));
    } catch (err: any) {
      setError(err.message || 'Failed to fetch user devices');
      setDevices([]);
    } finally {
      setLoading(false);
    }
  }, [user]);

  // Fetch bar chart data for all devices
  const fetchAllBarData = useCallback(async (devices: any[]) => {
    setLoading(true);
    setError(null);
    const newBarData: Record<string, { month: string; usage: number }[]> = {};
    for (const device of devices) {
      try {
        const response = await fetch('/api/postgres/device-history', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ deviceName: device.device_id })
        });
        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || 'Failed to fetch device history');
        }
        const data = await response.json();
        newBarData[device.device_id] = data.usage || [];
      } catch (error: any) {
        setError(error.message || 'Failed to fetch device history');
        newBarData[device.device_id] = [];
      }
    }
    setBarData(newBarData);
    setLoading(false);
  }, []);

  // Fetch all devices for the table
  const fetchAllDevices = useCallback(async () => {
    setTableLoading(true);
    setTableError(null);
    try {
      const response = await fetch('/api/postgres/all-devices');
      if (!response.ok) throw new Error('Failed to fetch all devices');
      const data = await response.json();
      setAllDevices(data.devices || []);
      setSystemStatus((prev) => ({
        ...prev,
        totalDevices: data.totalDevices ?? 0,
        activeDevices: data.activeDevices ?? 0,
        lastUpdate: new Date(),
      }));
    } catch (err: any) {
      setTableError(err.message || 'Failed to fetch all devices');
      setAllDevices([]);
      setSystemStatus((prev) => ({ ...prev, totalDevices: 0, activeDevices: 0 }));
    } finally {
      setTableLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchUserDevices();
    fetchAllDevices();
  }, [fetchUserDevices, fetchAllDevices]);

  useEffect(() => {
    if (devices.length > 0) {
      fetchAllBarData(devices);
        }
  }, [devices, fetchAllBarData]);
        
  // Subscribe to MQTT connection status
  useEffect(() => {
    let unsubscribe: (() => void) | undefined;
    loadMqttLibrary().then(() => {
      ensureMqttConnected(); // Ensure connection on dashboard load
      unsubscribe = subscribeMqttStatus((connected) => {
        setSystemStatus((prev) => ({ ...prev, mqttConnected: connected }));
      });
    });
    return () => {
      if (unsubscribe) unsubscribe();
    };
  }, []);

  // Devices Overview Table columns
  const tableHeaders = [
    { field: 'device_name', label: 'מספר שעון' },
    { field: 'device_type', label: 'סוג מכשיר' },
    { field: 'max_value', label: 'ערך מקסימלי' },
    { field: 'current_read', label: 'קריאה נוכחית' },
    { field: 'valve_state', label: 'סטטוס' },
    { field: 'battery_voltage', label: 'מתח סוללה' },
    // Optionally add threshold flags here
  ];

  // Sorting logic
  const sortedDevices = useMemo(() => {
    if (!allDevices || allDevices.length === 0) return [];
    const sorted = [...allDevices].sort((a, b) => {
      let valA = a[sortField];
      let valB = b[sortField];
      // Convert to number if possible
      if (!isNaN(Number(valA)) && valA !== null && valA !== undefined) valA = Number(valA);
      if (!isNaN(Number(valB)) && valB !== null && valB !== undefined) valB = Number(valB);
      if (valA === valB) return 0;
      if (valA === null || valA === undefined) return sortDirection === 'asc' ? 1 : -1;
      if (valB === null || valB === undefined) return sortDirection === 'asc' ? -1 : 1;
      if (typeof valA === 'number' && typeof valB === 'number') {
        return sortDirection === 'asc' ? valA - valB : valB - valA;
      }
      return sortDirection === 'asc'
        ? String(valA).localeCompare(String(valB))
        : String(valB).localeCompare(String(valA));
    });
    return sorted;
  }, [allDevices, sortField, sortDirection]);

  const handleSort = (field: string) => {
    if (sortField === field) {
      setSortDirection((prev) => (prev === 'asc' ? 'desc' : 'asc'));
    } else {
      setSortField(field);
      setSortDirection('asc');
    }
  };

  // Фильтрация устройств по поиску
  const filteredDevices = useMemo(() => {
    if (!search.trim()) return sortedDevices;
    const s = search.trim().toLowerCase();
    return sortedDevices.filter(device =>
      (device.f_name && device.f_name.toLowerCase().includes(s)) ||
      (device.s_name && device.s_name.toLowerCase().includes(s)) ||
      (device.device_id && device.device_id.toLowerCase().includes(s)) ||
      (device.device_name && device.device_name.toLowerCase().includes(s))
    );
  }, [sortedDevices, search]);

  // Для выпадающего списка на мобильных
  const filteredMobileDevices = useMemo(() => {
    if (!search.trim()) return allDevices;
    const s = search.trim().toLowerCase();
    return allDevices.filter(device =>
      (device.f_name && device.f_name.toLowerCase().includes(s)) ||
      (device.s_name && device.s_name.toLowerCase().includes(s)) ||
      (device.device_id && device.device_id.toLowerCase().includes(s)) ||
      (device.device_name && device.device_name.toLowerCase().includes(s))
    );
  }, [allDevices, search]);

  return (
    <div className="space-y-8">
      {/* System Status Card */}
      <SystemStatusCard status={systemStatus} />

      {/* Device Cards (GaugeChart + Bar Chart) */}
      {loading ? (
        <Skeleton className="w-full h-[400px]" />
      ) : error ? (
        <Alert variant="destructive"><AlertCircle className="h-4 w-4" /><AlertTitle>Error Loading Devices</AlertTitle><AlertDescription>{error}</AlertDescription></Alert>
      ) : devices.length === 0 ? (
        <p className="text-muted-foreground text-center py-4">No system devices found.</p>
      ) : (
        devices.map((device, idx) => (
          <div key={`${device.device_id || device.device_name || 'device'}_${idx}`} className="grid grid-cols-1 md:grid-cols-5 gap-6 mb-8">
            <Card className="col-span-1 md:col-span-2 w-full">
              <CardHeader>
                {/* Valve state above device id */}
                {(userRoleId === 0 || userRoleId === 1) && (
                  <div className="flex items-center gap-2 mb-2">
                    <ValveState deviceName={device.device_id} unit={device.device_type === 'Electric meter' ? 'kWh' : 'm³'} />
                  </div>
                )}
                <CardTitle>{t(device.device_type === 'Electric meter' ? 'gaugeChart.electricMeter' : 'gaugeChart.waterMeter', device.device_type)}: {device.device_id}</CardTitle>
                <p className="text-sm text-muted-foreground">{t('gaugeChart.devEUI', 'DEVEUI')}: {device.dev_eui || device.device_id}</p>
              </CardHeader>
              <CardContent>
                {(userRoleId === 0 || userRoleId === 1) ? (
                  <HorizontalBarChart
                    deviceName={device.device_id}
                    devEUI={device.dev_eui || device.device_id}
                    unit={device.device_type === 'Electric meter' ? 'kWh' : 'm³'}
                    deviceType={device.device_type}
                    hideValveState
                  />
                ) : (
                  <GaugeChart
                    deviceName={device.device_id}
                    devEUI={device.dev_eui || device.device_id}
                    unit={device.device_type === 'Electric meter' ? 'kWh' : 'm³'}
                  />
                )}
              </CardContent>
            </Card>
            <Card className="col-span-1 md:col-span-3 w-full">
                  <CardHeader>
                <CardTitle>{t('dashboard.usageTrend', 'Usage Trend (Monthly)')}</CardTitle>
                <p className="text-xs text-muted-foreground pt-1">{t('dashboard.device', 'Device')}: {device.device_id}</p>
                  </CardHeader>
                  <CardContent>
                {loading ? (
                  <Skeleton className="w-full" />
                ) : error ? (
                  <Alert variant="destructive">
                    <AlertCircle className="h-4 w-4" />
                    <AlertTitle>{t('errors.fetchingTrendData', 'Error Fetching Trend Data')}</AlertTitle>
                    <AlertDescription>{error}</AlertDescription>
                  </Alert>
                ) : (barData[device.device_id] && getProcessedBarData(barData[device.device_id], device.device_type).length > 0) ? (
                  <ResponsiveContainer width="100%" height={300}>
                    <AreaChart
                      data={getProcessedBarData(barData[device.device_id], device.device_type)}
                      margin={{
                        top: 5,
                        right: 0,
                        bottom: 60,
                        left: -10,
                      }}
                    >
                      <defs>
                        <linearGradient id="usageGradient" x1="0" y1="0" x2="0" y2="1">
                          <stop offset="0%" stopColor="#2196f3" stopOpacity={0.5} />
                          <stop offset="100%" stopColor="#2196f3" stopOpacity={0.1} />
                        </linearGradient>
                      </defs>
                      <CartesianGrid strokeDasharray="3 3" stroke="hsl(var(--border))" />
                      <XAxis
                        dataKey="label"
                        height={80}
                        axisLine={false}
                        tickLine={false}
                        tick={<VerticalTick />}
                      />
                      <YAxis tick={{ fontSize: 12 }} minTickGap={0} axisLine={false} tickLine={false} />
                      <Tooltip
                        content={<CustomUsageTooltip deviceType={device.device_type} />}
                        cursor={{ stroke: '#2196f3', strokeWidth: 1, opacity: 0.2 }}
                      />
                      <Area
                        type="monotone"
                        dataKey="usage"
                        stroke="#43a047"
                        strokeWidth={2}
                        fill="url(#usageGradient)"
                        dot={{ r: 3, stroke: '#43a047', strokeWidth: 2, fill: '#fff' }}
                        activeDot={{ r: 5 }}
                      />
                    </AreaChart>
                  </ResponsiveContainer>
                ) : (
                  <div className="flex items-center justify-center h-[300px] text-muted-foreground">
                    <div className="text-center">
                      <BellOff className="h-8 w-8 mx-auto mb-2 opacity-50" />
                      <p className="text-sm">{t('dashboard.noTrendData', 'No trend data available')}</p>
                    </div>
                  </div>
                )}
                  </CardContent>
                </Card>
        </div>
        ))
      )}

      {/* Поиск устройства */}
      <div className="w-full flex flex-col sm:flex-row gap-2 items-center mb-4" style={window.innerWidth === 768 ? { marginBottom: 80, width: '90vw', maxWidth: '90vw' } : window.innerWidth < 768 ? { width: '90vw', maxWidth: '90vw' } : {}}>
        <input
          type="text"
          className="w-full sm:w-72 border rounded px-3 py-2 text-sm"
          placeholder={t('dashboard.searchPlaceholder', 'Search by first name, last name or device number')}
          value={search}
          onChange={e => setSearch(e.target.value)}
        />
      </div>

      {/* Мобильный выпадающий список устройств */}
      <div className="block lg:hidden w-full mb-4" style={window.innerWidth === 768 ? { marginBottom: 80, width: '90vw', maxWidth: '90vw' } : window.innerWidth < 768 ? { width: '90vw', maxWidth: '90vw' } : {}}>
        <select
          className="w-full border rounded px-3 py-2 text-sm"
          value={selectedMobileDevice}
          onChange={e => {
            setSelectedMobileDevice(e.target.value);
            const dev = filteredMobileDevices.find(d => d.device_id === e.target.value || d.device_name === e.target.value);
            if (dev) {
              setSelectedDevice(dev);
              setModalOpen(true);
            }
          }}
        >
          <option value="">{t('dashboard.selectDevicePlaceholder', 'Select a device...')}</option>
          {filteredMobileDevices.map((device, idx) => (
            <option key={`${device.device_id || device.device_name || 'mobile'}_${idx}`} value={device.device_id || device.device_name}>
              {device.f_name || ''} {device.s_name || ''} ({device.device_id || device.device_name})
            </option>
          ))}
        </select>
      </div>

      {/* Таблица устройств только на lg+ */}
      <div className="hidden lg:block">
        <Card>
          <CardHeader>
            <CardTitle>{t('dashboard.devicesOverview', 'Devices Overview (Status from last 24 hours)')}</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="relative overflow-x-auto">
              {tableLoading ? (
                <Skeleton className="w-full h-[200px]" />
              ) : tableError ? (
                <Alert variant="destructive"><AlertCircle className="h-4 w-4" /><AlertTitle>Error Loading Devices</AlertTitle><AlertDescription>{tableError}</AlertDescription></Alert>
              ) : (
              <table dir="rtl" className="w-full text-sm text-left border-collapse mt-5">
                <thead>
                  <tr>
                    {tableHeaders.map((header) => (
                      <th
                        key={header.field}
                        className={"p-3 text-center border border-[#ddd] table-header cursor-pointer select-none custom-th-blue"}
                        onClick={() => handleSort(header.field)}
                      >
                        {header.label}
                        {sortField === header.field ? (sortDirection === 'asc' ? ' ▲' : ' ▼') : ''}
                      </th>
                    ))}
                  </tr>
                </thead>
                <tbody>
                    {filteredDevices.map((device, idx) => (
                      <tr key={`${device.device_name || device.device_id || 'row'}_${idx}`} className="border border-[#ddd] hover:bg-muted/50 transition-colors">
                        <td className="custom-td-cyan font-bold cursor-pointer text-center hover:underline" onClick={() => { setSelectedDevice(device); setModalOpen(true); }}>{device.device_name}</td>
                        <td className="custom-td-white text-center">{getDeviceTypeLabel(device.device_type)}</td>
                        <td className="custom-td-cyan font-bold text-center">
                          {device.max_value !== null && !isNaN(Number(device.max_value))
                            ? <span dir="ltr">{Number(device.max_value).toFixed(1)} {device.device_type === 'Electric meter' ? 'kWh' : 'm³'}</span>
                            : 'N/A'}
                        </td>
                        <td className="custom-td-white font-bold text-center">
                          {device.current_read !== null && !isNaN(Number(device.current_read))
                            ? <span dir="ltr">{Number(device.current_read).toFixed(3)} {device.device_type === 'Electric meter' ? 'kWh' : 'm³'}</span>
                            : 'N/A'}
                        </td>
                        <td className="custom-td-cyan text-center">
                          {(() => {
                            if (device.device_type && device.device_type.toLowerCase().includes('electric')) {
                              if (device.valve_state && device.valve_state.toLowerCase().includes('open')) {
                                return t('gaugeChart.on', 'On');
                              } else if (device.valve_state && device.valve_state.toLowerCase().includes('close')) {
                                return t('gaugeChart.off', 'Off');
                              } else {
                                return t('gaugeChart.noValve', 'No valve info');
                              }
                            } else {
                              if (device.valve_state && device.valve_state.toLowerCase().includes('open')) {
                                return t('gaugeChart.open', 'Open');
                              } else if (device.valve_state && device.valve_state.toLowerCase().includes('close')) {
                                return t('gaugeChart.close', 'Close');
                              } else {
                                return t('gaugeChart.noValve', 'No valve info');
                              }
                            }
                          })()}
                        </td>
                        <td className="custom-td-white text-center">
                          {device.battery_voltage !== null && !isNaN(Number(device.battery_voltage))
                            ? Number(device.battery_voltage).toFixed(2) + 'V'
                            : 'N/A'}
                        </td>
                      </tr>
                    ))}
                </tbody>
              </table>
              )}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Device Detail Modal */}
      {modalOpen && selectedDevice && (
        <DeviceDetailModal
          isOpen={modalOpen}
          onClose={() => setModalOpen(false)}
          device={{
            deviceId: selectedDevice.device_name,
            maxValue: selectedDevice.max_value ?? null,
            twentyFourHourValveState: selectedDevice.valve_state ?? null,
            twentyFourHourCurrentReading: selectedDevice.current_read ?? null,
            device_type: selectedDevice.device_type ?? null,
            roleId: typeof selectedDevice.roleId === 'number' ? selectedDevice.roleId : null,
            isAdminDevice: false,
            dev_eui: selectedDevice.dev_eui || '',
            // Add more fields as needed
          }}
          onMaxValueUpdate={() => fetchAllDevices()}
        />
      )}
    </div>
  );
};

export default SystemDashboardTab; 